/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Point;

import intrinsic.Function;
import intrinsic.mx.events.*;
import intrinsic.mx.controls.*;

public class Slider extends Control {
	int minimum, maximum, selection, thumb, increment, pageIncrement;

public Slider (Composite parent, int style) {
	super (parent, checkStyle (style));
}

public void addSelectionListener (SelectionListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener(listener);
	addListener (SWT.Selection,typedListener);
	addListener (SWT.DefaultSelection,typedListener);
}

static int checkStyle (int style) {
	return checkBits (style, SWT.HORIZONTAL, SWT.VERTICAL, 0, 0, 0, 0);
}

public Point computeSize (int wHint, int hHint, boolean changed) {
	checkWidget ();
	Point size = super.computeSize(wHint, hHint, changed);
	if (wHint == SWT.DEFAULT && hHint == SWT.DEFAULT) {
		if ((style & SWT.HORIZONTAL) != 0) {
			size.x = size.y * 10;
		} else {
			size.y = size.x * 10;
		}
	}
	return size;
}

void createHandle () {
	if ((style & SWT.HORIZONTAL) != 0) {
		object = new HScrollBar();
	} else {
		object = new VScrollBar();
	}
}

void createWidget() {
	super.createWidget ();
	setValues (0, 0, 100, 10, 1, 10);
}

public int getIncrement () {
	checkWidget ();
	return increment;
}

public int getMaximum () {
	checkWidget ();
	return maximum;
}

public int getMinimum () {
	checkWidget ();
	return minimum;
}

public int getPageIncrement () {
	checkWidget ();
	return pageIncrement;
}

public int getSelection () {
	checkWidget ();
	return selection;
}

public int getThumb () {
	checkWidget ();
	return thumb;
}

void hookEvents () {
	super.hookEvents ();
	object.addEventListener (ScrollEvent.SCROLL, scrollHandlerFunction ());
}

public void removeSelectionListener (SelectionListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);	
}
	
void scrollHandler(ScrollEvent event) {
	int detail = 0;
	if (event.detail == ScrollEventDetail.LINE_UP) {
		detail = SWT.ARROW_UP;
	} else if (event.detail == ScrollEventDetail.LINE_DOWN) {
		detail = SWT.ARROW_DOWN;
	} else if (event.detail == ScrollEventDetail.LINE_LEFT) {
		detail = SWT.ARROW_DOWN;
	} else if (event.detail == ScrollEventDetail.LINE_RIGHT) {
		detail = SWT.ARROW_UP;
	} else if (event.detail == ScrollEventDetail.PAGE_UP) {
		detail = SWT.PAGE_UP;
	} else if (event.detail == ScrollEventDetail.PAGE_DOWN) {
		detail = SWT.PAGE_DOWN;
	} else if (event.detail == ScrollEventDetail.PAGE_LEFT) {
		detail = SWT.PAGE_DOWN;
	} else if (event.detail == ScrollEventDetail.PAGE_RIGHT) {
		detail = SWT.PAGE_UP;
	} else if (event.detail == ScrollEventDetail.THUMB_TRACK) {
		detail = SWT.DRAG;
	} else if (event.detail == ScrollEventDetail.THUMB_POSITION) {
		detail = SWT.NONE;
	} else if (event.detail == ScrollEventDetail.AT_TOP) {
		detail = SWT.HOME;
	} else if (event.detail == ScrollEventDetail.AT_BOTTOM) {
		detail = SWT.END;
	} else if (event.detail == ScrollEventDetail.AT_LEFT) {
		detail = SWT.HOME;
	} else if (event.detail == ScrollEventDetail.AT_RIGHT) {
		detail = SWT.END;
	}
	sendSelection(detail, (int)event.position);
}

native Function scrollHandlerFunction ()/*{
	return scrollHandler__Lmx_events_ScrollEvent_2;
}*/;

void sendSelection (int detail, int selection) {
	this.selection = selection;
	Event event = new Event ();
	event.detail = detail;
	sendEvent (SWT.Selection, event);
}

public void setIncrement (int value) {
	setValues(selection, minimum, maximum, thumb, value, pageIncrement);
}

public void setMaximum (int value) {
	setValues(selection, minimum, value, thumb, increment, pageIncrement);
}

public void setMinimum (int value) {
	setValues(selection, value, maximum, thumb, increment, pageIncrement);
}

public void setPageIncrement (int value) {
	setValues(selection, minimum, maximum, thumb, increment, value);
}

public void setSelection (int value) {
	setValues(value, minimum, maximum, thumb, increment, pageIncrement);
}

public void setThumb (int value) {
	setValues(selection, minimum, maximum, value, increment, pageIncrement);
}

public void setValues (int selection, int minimum, int maximum, int thumb, int increment, int pageIncrement) {
	checkWidget();
	if (minimum < 0) return;
	if (maximum < 0) return;
	if (thumb < 1) return;
	if (increment < 1) return;
	if (pageIncrement < 1) return;
	//TODO - more range checks
	this.selection = selection;
	this.minimum = minimum;
	this.maximum = maximum;
	this.thumb = thumb;
	this.increment = increment;
	this.pageIncrement = pageIncrement;
	intrinsic.mx.controls.scrollClasses.ScrollBar bar = (intrinsic.mx.controls.scrollClasses.ScrollBar)object;
	bar.minScrollPosition = minimum;
	bar.maxScrollPosition = maximum - thumb;
	bar.scrollPosition = selection;
	bar.pageSize = thumb;
	bar.lineScrollSize = increment;
	bar.pageScrollSize = pageIncrement;
}

}